// Heat Monitoring JavaScript Functions

document.addEventListener('DOMContentLoaded', function() {
    initializeHeatMonitoring();
    setupEventListeners();
    startAutoRefresh();
});

// Global variables
let autoRefreshInterval;
let isAutoRefreshEnabled = true;
let lastUpdateTime = new Date();

// Initialize heat monitoring dashboard
function initializeHeatMonitoring() {
    console.log('Heat Monitoring System initialized');
    updateLastRefreshTime();
    setupTooltips();
    animateChartBars();
}

// Setup event listeners
function setupEventListeners() {
    // Menu toggle functionality (from main dashboard)
    const menuToggle = document.getElementById('menuToggle');
    const sidebar = document.getElementById('sidebar');
    
    if (menuToggle && sidebar) {
        menuToggle.addEventListener('click', function() {
            sidebar.classList.toggle('open');
        });
    }

    // Auto-refresh toggle
    const refreshIndicator = document.getElementById('refreshIndicator');
    if (refreshIndicator) {
        refreshIndicator.addEventListener('click', toggleAutoRefresh);
    }

    // Chart bar hover effects
    const chartBars = document.querySelectorAll('.bar');
    chartBars.forEach(bar => {
        bar.addEventListener('mouseenter', function() {
            this.style.transform = 'scaleY(1.05)';
        });
        
        bar.addEventListener('mouseleave', function() {
            this.style.transform = 'scaleY(1)';
        });
    });

    // Weather stat hover effects
    const weatherStats = document.querySelectorAll('.weather-stat');
    weatherStats.forEach(stat => {
        stat.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-2px)';
        });
        
        stat.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });
}


// Setup tooltips
function setupTooltips() {
    const tooltipElements = document.querySelectorAll('[data-tooltip]');
    
    tooltipElements.forEach(element => {
        element.classList.add('tooltip');
    });
}

// Animate chart bars on load
function animateChartBars() {
    const bars = document.querySelectorAll('.bar');
    
    bars.forEach((bar, index) => {
        setTimeout(() => {
            bar.style.animation = 'fadeInUp 0.5s ease forwards';
        }, index * 100);
    });
}

// Export data functionality
function exportData() {
    console.log('Exporting heat monitoring data...');
    showSuccessMessage('Generating heat monitoring report...');
    
    // In a real application, this would generate and download a report
    setTimeout(() => {
        // Simulate file download
        const link = document.createElement('a');
        link.href = '#';
        link.download = `heat-monitoring-report-${new Date().toISOString().split('T')[0]}.pdf`;
        link.click();
        
        showSuccessMessage('Report exported successfully');
    }, 2000);
}

// View history functionality
function viewHistory() {
    console.log('Opening heat index history...');
    // This would typically open a modal or navigate to a history page
    alert('Heat Index History\n\nThis would open a detailed view of historical heat index data, including:\n• Daily/weekly/monthly trends\n• Peak heat periods\n• Location comparisons\n• Incident correlations');
}

// Trigger manual alert
function triggerAlert() {
    const confirmation = confirm('Are you sure you want to trigger a manual heat alert?\n\nThis will notify all supervisors and emergency teams.');
    
    if (confirmation) {
        console.log('Manual heat alert triggered');
        showSuccessMessage('Emergency heat alert has been sent to all supervisors');
        
        // Add new alert to the alerts list
        addNewAlert('Manual Alert Triggered', 'Emergency heat alert sent to all supervisors', 'danger', 'just now');
    }
}

// CSS animations for fade in up effect
const style = document.createElement('style');
style.textContent = `
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(20px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
`;
document.head.appendChild(style);