<?php
$page_title = "Schedule Management - HeatGuard";
$additional_css = ['assets/css/schedule-management.css'];
$additional_js = ['assets/js/schedule-management.js'];

$user = [
    'name' => 'Officer Jerome Buntalidad',
    'role' => 'Traffic Enforcer',
    'department' => 'Bacolod TMU',
    'badge' => 'TMU-2024-001',
    'status' => 'on_duty'
];

// Current heat index for schedule recommendations
$current_heat_index = 42;
$heat_level = 'DANGER';

// Sample officer data
$officers = [
    [
        'id' => 1,
        'name' => 'Officer Jerome Buntalidad',
        'badge' => 'TMU-2024-001',
        'position' => 'City Hall Intersection',
        'shift_start' => '06:00',
        'shift_end' => '14:00',
        'status' => 'on_duty',
        'exposure_time' => 180, // minutes
        'last_break' => '10:30',
        'next_break' => '12:30',
        'heat_exposure_level' => 'high'
    ],
    [
        'id' => 2,
        'name' => 'Officer Maria Santos',
        'badge' => 'TMU-2024-002',
        'position' => 'Public Plaza',
        'shift_start' => '06:00',
        'shift_end' => '14:00',
        'status' => 'on_break',
        'exposure_time' => 160,
        'last_break' => '11:45',
        'next_break' => '13:45',
        'heat_exposure_level' => 'medium'
    ],
    [
        'id' => 3,
        'name' => 'Officer Roberto Cruz',
        'badge' => 'TMU-2024-003',
        'position' => 'Main Street',
        'shift_start' => '14:00',
        'shift_end' => '22:00',
        'status' => 'scheduled',
        'exposure_time' => 0,
        'last_break' => null,
        'next_break' => '16:00',
        'heat_exposure_level' => 'low'
    ],
    [
        'id' => 4,
        'name' => 'Officer Ana Reyes',
        'badge' => 'TMU-2024-004',
        'position' => 'School Zone',
        'shift_start' => '14:00',
        'shift_end' => '22:00',
        'status' => 'scheduled',
        'exposure_time' => 0,
        'last_break' => null,
        'next_break' => '16:00',
        'heat_exposure_level' => 'low'
    ]
];

// Today's schedule
$todays_schedule = [
    [
        'time' => '06:00 - 14:00',
        'shift' => 'Morning Shift',
        'officers' => ['Jerome Buntalidad', 'Maria Santos'],
        'positions' => ['City Hall', 'Public Plaza'],
        'status' => 'active'
    ],
    [
        'time' => '14:00 - 22:00',
        'shift' => 'Afternoon Shift',
        'officers' => ['Roberto Cruz', 'Ana Reyes'],
        'positions' => ['Main Street', 'School Zone'],
        'status' => 'scheduled'
    ],
    [
        'time' => '22:00 - 06:00',
        'shift' => 'Night Shift',
        'officers' => ['Carlos Mendoza', 'Lisa Garcia'],
        'positions' => ['City Hall', 'Main Street'],
        'status' => 'scheduled'
    ]
];

// Pending alerts and recommendations
$schedule_alerts = [
    [
        'type' => 'break_overdue',
        'officer' => 'Jerome Buntalidad',
        'message' => 'Break overdue - High heat exposure detected',
        'time' => '5 min ago',
        'priority' => 'high'
    ],
    [
        'type' => 'rotation_needed',
        'officer' => 'Maria Santos',
        'message' => 'Rotation recommended - 3 hours exposure',
        'time' => '10 min ago',
        'priority' => 'medium'
    ],
    [
        'type' => 'schedule_conflict',
        'officer' => 'Roberto Cruz',
        'message' => 'Afternoon shift adjustment needed',
        'time' => '1 hr ago',
        'priority' => 'low'
    ]
];

// Break schedule recommendations based on heat index
function getBreakRecommendation($heat_index) {
    if ($heat_index >= 40) {
        return [
            'interval' => 60, // minutes
            'duration' => 15, // minutes
            'location' => 'Air-conditioned area required'
        ];
    } elseif ($heat_index >= 32) {
        return [
            'interval' => 90,
            'duration' => 10,
            'location' => 'Shaded area recommended'
        ];
    } else {
        return [
            'interval' => 120,
            'duration' => 10,
            'location' => 'Standard break area'
        ];
    }
}

function getStatusClass($status) {
    switch($status) {
        case 'on_duty': return 'status-active';
        case 'on_break': return 'status-break';
        case 'off_duty': return 'status-offline';
        case 'scheduled': return 'status-scheduled';
        default: return 'status-offline';
    }
}

function getExposureClass($level) {
    switch($level) {
        case 'high': return 'exposure-high';
        case 'medium': return 'exposure-medium';
        case 'low': return 'exposure-low';
        default: return 'exposure-low';
    }
}

function getPriorityClass($priority) {
    switch($priority) {
        case 'high': return 'priority-high';
        case 'medium': return 'priority-medium';
        case 'low': return 'priority-low';
        default: return 'priority-low';
    }
}

$break_recommendation = getBreakRecommendation($current_heat_index);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?></title>
    <link rel="stylesheet" href="assets/css/dashboard.css">
    <link rel="stylesheet" href="assets/css/schedule-management.css">
    <link rel="stylesheet" href="assets/css/main-footer.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
</head>
<body>
    <div class="dashboard-container">
        <!-- Sidebar -->
        <aside class="sidebar" id="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">
                    <span class="logo-icon">🚦</span>
                    <span>HeatGuard</span>
                </div>
            </div>
            
            <nav class="sidebar-nav">
                <div class="nav-section">
                    <div class="nav-section-title">Main Dashboard</div>
                    <div class="nav-item">
                        <a href="dashboard.php" class="nav-link">
                            <i class="nav-icon fas fa-tachometer-alt"></i>
                            <span>Overview</span>
                        </a>
                    </div>
                </div>

                <div class="nav-section">
                    <div class="nav-section-title">Core Modules</div>
                    <div class="nav-item">
                        <a href="user-management.php" class="nav-link">
                            <i class="nav-icon fas fa-users"></i>
                            <span>User Management</span>
                        </a>
                    </div>
                    <div class="nav-item">
                        <a href="heat-monitoring.php" class="nav-link">
                            <i class="nav-icon fas fa-thermometer-half"></i>
                            <span>Heat Index Monitoring</span>
                        </a>
                    </div>
                    <div class="nav-item">
                        <a href="schedule-management.php" class="nav-link active">
                            <i class="nav-icon fas fa-calendar-alt"></i>
                            <span>Schedule Management</span>
                        </a>
                    </div>
                    <div class="nav-item">
                        <a href="safety-guidelines.php" class="nav-link">
                            <i class="nav-icon fas fa-shield-alt"></i>
                            <span>Safety Guidelines</span>
                        </a>
                    </div>
                    <div class="nav-item">
                        <a href="emergency-response.php" class="nav-link">
                            <i class="nav-icon fas fa-exclamation-triangle"></i>
                            <span>Emergency Response</span>
                        </a>
                    </div>
                    <div class="nav-item">
                        <a href="reporting-system.php" class="nav-link">
                            <i class="nav-icon fas fa-chart-bar"></i>
                            <span>Automated Reports</span>
                        </a>
                    </div>
                </div>

                <div class="nav-section">
                    <div class="nav-section-title">Quick Actions</div>
                    <div class="nav-item">
                        <a href="#" class="nav-link" onclick="triggerEmergency()">
                            <i class="nav-icon fas fa-ambulance"></i>
                            <span>Emergency Alert</span>
                        </a>
                    </div>
                    <div class="nav-item">
                        <a href="#" class="nav-link">
                            <i class="nav-icon fas fa-file-download"></i>
                            <span>Export Reports</span>
                        </a>
                    </div>
                </div>

                <div class="nav-section">
                    <div class="nav-section-title">Account</div>
                    <div class="nav-item">
                        <a href="profile.php" class="nav-link">
                            <i class="nav-icon fas fa-user"></i>
                            <span>Profile Settings</span>
                        </a>
                    </div>
                    <div class="nav-item">
                        <a href="index.php" class="nav-link">
                            <i class="nav-icon fas fa-sign-out-alt"></i>
                            <span>Logout</span>
                        </a>
                    </div>
                </div>
            </nav>
        </aside>

        <!-- Main Content -->
        <main class="main-content">
            <!-- Top Bar -->
            <header class="top-bar">
                <div class="top-bar-left">
                    <button class="menu-toggle" id="menuToggle">
                        <i class="fas fa-bars"></i>
                    </button>
                    <h1 class="page-title">Schedule Management</h1>
                </div>
                
                <div class="top-bar-right">
                    <div class="heat-indicator <?php echo $heat_level === 'DANGER' ? 'heat-danger' : ($heat_level === 'CAUTION' ? 'heat-caution' : 'heat-safe'); ?>">
                        <i class="fas fa-thermometer-half"></i>
                        <span><?php echo $current_heat_index; ?>°C - <?php echo $heat_level; ?></span>
                    </div>
                    
                    <div class="status-indicator <?php echo $user['status'] === 'on_duty' ? 'status-online' : 'status-offline'; ?>">
                        <i class="fas fa-circle"></i>
                        <?php echo $user['status'] === 'on_duty' ? 'On Duty' : 'Off Duty'; ?>
                    </div>
                    
                    <div class="user-menu">
                        <div class="user-avatar">
                            <?php echo strtoupper(substr($user['name'], 8, 1)) . strtoupper(substr($user['name'], 13, 1)); ?>
                        </div>
                        <div class="user-info">
                            <h4><?php echo explode(' ', $user['name'])[1] . ' ' . explode(' ', $user['name'])[2]; ?></h4>
                            <p><?php echo $user['role']; ?></p>
                        </div>
                    </div>
                </div>
            </header>

            <!-- Schedule Management Content -->
            <div class="schedule-management-content">
                
                <!-- Heat-Based Recommendations -->
                <div class="recommendations-section">
                    <div class="recommendations-card <?php echo $heat_level === 'DANGER' ? 'danger-alert' : ''; ?>">
                        <div class="card-header">
                            <div class="card-title">
                                <i class="fas fa-lightbulb"></i>
                                Heat-Based Schedule Recommendations
                            </div>
                            <div class="heat-level-badge <?php echo $heat_level === 'DANGER' ? 'status-danger' : ($heat_level === 'CAUTION' ? 'status-caution' : 'status-safe'); ?>">
                                <?php echo $heat_level; ?>
                            </div>
                        </div>
                        <div class="recommendations-grid">
                            <div class="recommendation-item">
                                <div class="rec-icon break-icon">
                                    <i class="fas fa-clock"></i>
                                </div>
                                <div class="rec-info">
                                    <h4>Break Interval</h4>
                                    <p>Every <?php echo $break_recommendation['interval']; ?> minutes</p>
                                </div>
                            </div>
                            <div class="recommendation-item">
                                <div class="rec-icon duration-icon">
                                    <i class="fas fa-pause"></i>
                                </div>
                                <div class="rec-info">
                                    <h4>Break Duration</h4>
                                    <p><?php echo $break_recommendation['duration']; ?> minutes minimum</p>
                                </div>
                            </div>
                            <div class="recommendation-item">
                                <div class="rec-icon location-icon">
                                    <i class="fas fa-map-marker-alt"></i>
                                </div>
                                <div class="rec-info">
                                    <h4>Break Location</h4>
                                    <p><?php echo $break_recommendation['location']; ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Main Dashboard Grid -->
                <div class="dashboard-grid">
                    <!-- Current Officers Status -->
                    <div class="dashboard-card officers-card">
                        <div class="card-header">
                            <h3>
                                <i class="fas fa-users"></i>
                                Officers Status
                            </h3>
                            <button class="btn-add" onclick="addNewOfficer()">
                                <i class="fas fa-plus"></i>
                                Add Officer
                            </button>
                        </div>
                        <div class="officers-list">
                            <?php foreach($officers as $officer): ?>
                            <div class="officer-item" data-officer-id="<?php echo $officer['id']; ?>">
                                <div class="officer-avatar">
                                    <?php echo strtoupper(substr($officer['name'], 8, 1)) . strtoupper(substr($officer['name'], strrpos($officer['name'], ' ') + 1, 1)); ?>
                                </div>
                                <div class="officer-info">
                                    <h4><?php echo $officer['name']; ?></h4>
                                    <p><?php echo $officer['badge']; ?> • <?php echo $officer['position']; ?></p>
                                    <div class="officer-schedule">
                                        <span><?php echo $officer['shift_start']; ?> - <?php echo $officer['shift_end']; ?></span>
                                    </div>
                                </div>
                                <div class="officer-stats">
                                    <div class="exposure-time <?php echo getExposureClass($officer['heat_exposure_level']); ?>">
                                        <i class="fas fa-thermometer-half"></i>
                                        <?php echo floor($officer['exposure_time'] / 60); ?>h <?php echo $officer['exposure_time'] % 60; ?>m
                                    </div>
                                    <div class="officer-status <?php echo getStatusClass($officer['status']); ?>">
                                        <?php echo ucfirst(str_replace('_', ' ', $officer['status'])); ?>
                                    </div>
                                </div>
                                <div class="officer-actions">
                                    <button class="btn-break" onclick="sendToBreak(<?php echo $officer['id']; ?>)" <?php echo $officer['status'] !== 'on_duty' ? 'disabled' : ''; ?>>
                                        <i class="fas fa-pause"></i>
                                    </button>
                                    <button class="btn-edit" onclick="editOfficer(<?php echo $officer['id']; ?>)">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>

                    <!-- Today's Schedule -->
                    <div class="dashboard-card schedule-card">
                        <div class="card-header">
                            <h3>
                                <i class="fas fa-calendar-day"></i>
                                Today's Schedule
                            </h3>
                            <button class="btn-add" onclick="editSchedule()">
                                <i class="fas fa-edit"></i>
                                Edit Schedule
                            </button>
                        </div>
                        <div class="schedule-timeline">
                            <?php foreach($todays_schedule as $schedule): ?>
                            <div class="schedule-item <?php echo $schedule['status']; ?>">
                                <div class="schedule-time">
                                    <strong><?php echo $schedule['time']; ?></strong>
                                    <span class="shift-name"><?php echo $schedule['shift']; ?></span>
                                </div>
                                <div class="schedule-details">
                                    <div class="assigned-officers">
                                        <?php foreach($schedule['officers'] as $officer): ?>
                                        <span class="officer-tag"><?php echo $officer; ?></span>
                                        <?php endforeach; ?>
                                    </div>
                                    <div class="assigned-positions">
                                        <?php echo implode(' • ', $schedule['positions']); ?>
                                    </div>
                                </div>
                                <div class="schedule-status">
                                    <span class="status-badge <?php echo $schedule['status']; ?>">
                                        <?php echo ucfirst($schedule['status']); ?>
                                    </span>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>

                    <!-- Schedule Alerts -->
                    <div class="dashboard-card alerts-card">
                        <div class="card-header">
                            <h3>
                                <i class="fas fa-bell"></i>
                                Schedule Alerts
                            </h3>
                            <span class="alert-count"><?php echo count($schedule_alerts); ?> pending</span>
                        </div>
                        <div class="alerts-list">
                            <?php foreach($schedule_alerts as $alert): ?>
                            <div class="alert-item <?php echo getPriorityClass($alert['priority']); ?>">
                                <div class="alert-icon">
                                    <i class="fas <?php echo $alert['type'] === 'break_overdue' ? 'fa-clock' : ($alert['type'] === 'rotation_needed' ? 'fa-sync-alt' : 'fa-calendar-times'); ?>"></i>
                                </div>
                                <div class="alert-info">
                                    <h5><?php echo $alert['officer']; ?></h5>
                                    <p><?php echo $alert['message']; ?></p>
                                    <span class="alert-time"><?php echo $alert['time']; ?></span>
                                </div>
                                <div class="alert-actions">
                                    <button class="btn-resolve" onclick="resolveAlert('<?php echo $alert['type']; ?>', '<?php echo $alert['officer']; ?>')">
                                        Resolve
                                    </button>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>

                    <!-- Break Management -->
                    <div class="dashboard-card breaks-card">
                        <div class="card-header">
                            <h3>
                                <i class="fas fa-pause-circle"></i>
                                Break Management
                            </h3>
                        </div>
                        <div class="breaks-summary">
                            <div class="break-stat">
                                <div class="stat-value">2</div>
                                <div class="stat-label">Currently on Break</div>
                            </div>
                            <div class="break-stat">
                                <div class="stat-value">5</div>
                                <div class="stat-label">Breaks Due Soon</div>
                            </div>
                            <div class="break-stat">
                                <div class="stat-value"><?php echo $break_recommendation['interval']; ?>min</div>
                                <div class="stat-label">Recommended Interval</div>
                            </div>
                        </div>
                        <div class="break-actions">
                            <button class="action-button action-primary" onclick="scheduleBreaks()">
                                <i class="fas fa-calendar-plus"></i>
                                Schedule Breaks
                            </button>
                            <button class="action-button action-secondary" onclick="viewBreakHistory()">
                                <i class="fas fa-history"></i>
                                Break History
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Action Buttons -->
                <div class="schedule-actions">
                    <button class="action-button action-primary" onclick="autoSchedule()">
                        <i class="fas fa-magic"></i>
                        Auto-Schedule Based on Heat
                    </button>
                    <button class="action-button action-secondary" onclick="exportSchedule()">
                        <i class="fas fa-download"></i>
                        Export Schedule
                    </button>
                    <button class="action-button action-primary" onclick="sendBulkBreaks()">
                        <i class="fas fa-pause"></i>
                        Send All to Break
                    </button>
                    <button class="action-button action-danger" onclick="emergencyRotation()">
                        <i class="fas fa-exclamation-triangle"></i>
                        Emergency Rotation
                    </button>
                </div>
            </div>
        </main>
    </div>

    <!-- Add Officer Modal -->
    <div class="modal" id="addOfficerModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Add New Officer</h3>
                <button class="modal-close" onclick="closeModal('addOfficerModal')">&times;</button>
            </div>
            <div class="modal-body">
                <form id="addOfficerForm">
                    <div class="form-group">
                        <label for="officerName">Officer Name</label>
                        <input type="text" id="officerName" name="officerName" required>
                    </div>
                    <div class="form-group">
                        <label for="badgeNumber">Badge Number</label>
                        <input type="text" id="badgeNumber" name="badgeNumber" required>
                    </div>
                    <div class="form-group">
                        <label for="position">Assigned Position</label>
                        <select id="position" name="position" required>
                            <option value="">Select Position</option>
                            <option value="City Hall">City Hall Intersection</option>
                            <option value="Public Plaza">Public Plaza</option>
                            <option value="Main Street">Main Street</option>
                            <option value="School Zone">School Zone</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="shiftStart">Shift Start</label>
                        <input type="time" id="shiftStart" name="shiftStart" required>
                    </div>
                    <div class="form-group">
                        <label for="shiftEnd">Shift End</label>
                        <input type="time" id="shiftEnd" name="shiftEnd" required>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button class="btn-cancel" onclick="closeModal('addOfficerModal')">Cancel</button>
                <button class="btn-save" onclick="saveNewOfficer()">Add Officer</button>
            </div>
        </div>
    </div>

    <script src="assets/js/dashboard.js"></script>
    <script src="assets/js/schedule-management.js"></script>
     <?php include 'main-footer.php'; ?>
</body>
</html>